<?php

namespace Plugins\SAMBans;

use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Config;

class Plugin
{
    protected $pluginPath;
    protected $manifest;
    protected $config;

    public function __construct($pluginPath, $manifest)
    {
        $this->pluginPath = $pluginPath;
        $this->manifest = $manifest;
        $this->loadConfig();
    }

    protected function loadConfig()
    {
        $configPath = $this->pluginPath . '/config.json';
        if (file_exists($configPath)) {
            $this->config = json_decode(file_get_contents($configPath), true) ?: [];
        } else {
            $this->config = $this->getDefaultConfig();
        }
    }

    protected function getDefaultConfig()
    {
        return [
            'database' => [
                'host' => env('SAM_DB_HOST', 'localhost'),
                'port' => env('SAM_DB_PORT', 3306),
                'database' => env('SAM_DB_DATABASE', 'sambans'),
                'username' => env('SAM_DB_USERNAME', 'sambans'),
                'password' => env('SAM_DB_PASSWORD', '')
            ],
            'display' => [
                'per_page' => 25,
                'show_expired' => true,
                'show_unbanned' => false
            ]
        ];
    }

    public function onInstall()
    {
        Log::info('SAM Bans plugin installed successfully');
        
        // Create default config file
        $configPath = $this->pluginPath . '/config.json';
        if (!file_exists($configPath)) {
            file_put_contents($configPath, json_encode($this->getDefaultConfig(), JSON_PRETTY_PRINT));
        }

        return true;
    }

    public function onActivate()
    {
        Log::info('SAM Bans plugin activated');
        
        // Set up the external database connection
        $this->setupExternalDatabase();
        
        // Test the connection
        if (!$this->testDatabaseConnection()) {
            Log::warning('SAM Bans: Could not connect to external database. Please check configuration.');
        }

        return true;
    }

    public function onDeactivate()
    {
        Log::info('SAM Bans plugin deactivated');
        return true;
    }

    public function onUninstall()
    {
        Log::info('SAM Bans plugin uninstalled');
        return true;
    }

    protected function setupExternalDatabase()
    {
        // Add the SAM database connection to Laravel's config
        Config::set('database.connections.sambans', [
            'driver' => 'mysql',
            'host' => $this->config['db_host'] ?? 'localhost',
            'port' => $this->config['db_port'] ?? 3306,
            'database' => $this->config['db_database'] ?? 'sambans',
            'username' => $this->config['db_username'] ?? 'sambans',
            'password' => $this->config['db_password'] ?? '',
            'charset' => 'utf8mb4',
            'collation' => 'utf8mb4_unicode_ci',
            'prefix' => '',
            'strict' => false,
            'engine' => null,
        ]);
    }

    protected function testDatabaseConnection()
    {
        try {
            DB::connection('sambans')->getPdo();
            Log::info('SAM Bans: Successfully connected to external database');
            return true;
        } catch (\Exception $e) {
            Log::error('SAM Bans: Database connection failed: ' . $e->getMessage());
            return false;
        }
    }

    public function getConfig()
    {
        return $this->config;
    }

    public function updateConfig($newConfig)
    {
        $this->config = array_merge($this->config, $newConfig);
        $configPath = $this->pluginPath . '/config.json';
        file_put_contents($configPath, json_encode($this->config, JSON_PRETTY_PRINT));
        
        // Update the database connection
        $this->setupExternalDatabase();
        
        return true;
    }
}